const fs = require('fs');
const path = require('path');
const os = require('os');

module.exports = {
  /**
   * 获取依赖列表
   */
  getDependencies() {
    return {
      'open': '^10.0.0'  // 使用open包打开浏览器
    };
  },

  /**
   * 获取元信息
   */
  getMetadata() {
    return {
      name: 'mermaid-viewer',
      description: '将Mermaid图表代码渲染成可视化图形，支持单图和画布模式',
      version: '1.2.0',
      category: 'visualization',
      author: '鲁班',
      tags: ['mermaid', 'diagram', 'visualization', 'graph', 'canvas'],
      manual: '@manual://mermaid-viewer'
    };
  },

  /**
   * 获取参数Schema
   */
  getSchema() {
    return {
      type: 'object',
      properties: {
        // 单图模式参数
        mermaidCode: {
          type: 'string',
          description: 'Mermaid图表代码（单图模式）'
        },
        title: {
          type: 'string',
          description: '图表/画布标题',
          default: 'Mermaid Diagram'
        },
        description: {
          type: 'string',
          description: '图表/画布的描述说明',
          default: ''
        },
        
        // 画布模式参数
        canvasId: {
          type: 'string',
          description: '画布ID，传入则进入画布模式'
        },
        charts: {
          type: 'array',
          description: '画布模式下的图表数组',
          items: {
            type: 'object',
            properties: {
              title: { type: 'string' },
              description: { type: 'string' },
              mermaidCode: { type: 'string' }
            }
          }
        },
        
        // 通用参数
        theme: {
          type: 'string',
          description: '图表主题',
          enum: ['default', 'dark', 'forest', 'neutral'],
          default: 'default'
        }
      }
    };
  },

  /**
   * 验证参数
   */
  validate(params) {
    const errors = [];
    
    // 检查模式
    const isCanvasMode = params.canvasId || params.charts;
    const isSingleMode = params.mermaidCode;
    
    if (!isCanvasMode && !isSingleMode) {
      errors.push('必须提供mermaidCode（单图模式）或canvasId/charts（画布模式）');
    }
    
    if (isCanvasMode && isSingleMode) {
      errors.push('不能同时使用单图和画布模式');
    }
    
    if (params.charts && !Array.isArray(params.charts)) {
      errors.push('charts必须是数组');
    }
    
    if (params.theme && !['default', 'dark', 'forest', 'neutral'].includes(params.theme)) {
      errors.push('theme 必须是 default/dark/forest/neutral 之一');
    }
    
    return {
      valid: errors.length === 0,
      errors: errors
    };
  },

  /**
   * 执行工具
   */
  async execute(params) {
    try {
      // 验证参数
      const validation = this.validate(params);
      if (!validation.valid) {
        throw new Error(`参数验证失败: ${validation.errors.join(', ')}`);
      }
      
      // 判断模式
      const isCanvasMode = params.canvasId || params.charts;
      
      if (isCanvasMode) {
        return await this.executeCanvasMode(params);
      } else {
        return await this.executeSingleMode(params);
      }
      
    } catch (error) {
      return {
        success: false,
        error: {
          code: error.code || 'EXECUTION_ERROR',
          message: error.message,
          details: error.stack
        }
      };
    }
  },

  /**
   * 单图模式执行
   */
  async executeSingleMode(params) {
    const title = params.title || 'Mermaid Diagram';
    const description = params.description || '';
    const theme = params.theme || 'default';
    const mermaidCode = params.mermaidCode;
    
    // 生成单图HTML
    const htmlContent = this.generateSingleHTML(title, description, mermaidCode, theme);
    
    // 保存文件
    const timestamp = Date.now();
    const fileName = `mermaid_${timestamp}.html`;
    const filePath = await this.saveHTML(fileName, htmlContent);
    
    // 打开浏览器
    const open = await loadModule('open');
    await open(filePath);
    
    return {
      success: true,
      data: {
        mode: 'single',
        filePath: filePath,
        opened: true,
        timestamp: new Date().toISOString(),
        message: `图表已生成并在浏览器中打开: ${fileName}`
      }
    };
  },

  /**
   * 画布模式执行
   */
  async executeCanvasMode(params) {
    const canvasId = params.canvasId || `canvas_${Date.now()}`;
    const title = params.title || '图表画布';
    const description = params.description || '';
    const theme = params.theme || 'default';
    const charts = params.charts || [];
    
    // 生成画布HTML
    const htmlContent = this.generateCanvasHTML(canvasId, title, description, charts, theme);
    
    // 保存文件（使用canvasId作为文件名）
    const fileName = `${canvasId}.html`;
    const filePath = await this.saveHTML(fileName, htmlContent);
    
    // 打开浏览器
    const open = await loadModule('open');
    await open(filePath);
    
    return {
      success: true,
      data: {
        mode: 'canvas',
        canvasId: canvasId,
        totalCharts: charts.length,
        filePath: filePath,
        opened: true,
        timestamp: new Date().toISOString(),
        message: `画布已生成（包含${charts.length}个图表）: ${fileName}`
      }
    };
  },

  /**
   * 生成单图HTML
   */
  generateSingleHTML(title, description, mermaidCode, theme) {
    // 对mermaid代码进行预处理
    const processedCode = this.processMermaidCode(mermaidCode);
    
    return `<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>${this.escapeHtml(title)}</title>
    <script src="https://cdn.jsdelivr.net/npm/mermaid@10/dist/mermaid.min.js"></script>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
            margin: 0;
            padding: 20px;
            background: ${theme === 'dark' ? '#1e1e1e' : '#ffffff'};
            color: ${theme === 'dark' ? '#e0e0e0' : '#333333'};
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
        }
        h1 {
            margin-bottom: 10px;
            font-size: 24px;
            color: ${theme === 'dark' ? '#ffffff' : '#2c3e50'};
        }
        .description {
            margin-bottom: 20px;
            padding: 10px;
            background: ${theme === 'dark' ? '#2d2d2d' : '#f5f5f5'};
            border-radius: 5px;
            line-height: 1.6;
        }
        .mermaid-container {
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 400px;
            padding: 20px;
            background: ${theme === 'dark' ? '#252525' : '#fafafa'};
            border-radius: 8px;
            border: 1px solid ${theme === 'dark' ? '#404040' : '#e0e0e0'};
            overflow: auto;
        }
        .mermaid {
            width: 100%;
            height: auto;
        }
        .error-message {
            color: #ff5252;
            padding: 20px;
            background: #ffebee;
            border-radius: 5px;
            margin: 20px 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>${this.escapeHtml(title)}</h1>
        ${description ? `<div class="description">${this.escapeHtml(description)}</div>` : ''}
        <div class="mermaid-container">
            <pre class="mermaid">
${processedCode}
            </pre>
        </div>
    </div>
    <script>
        mermaid.initialize({ 
            startOnLoad: true,
            theme: '${theme}',
            securityLevel: 'loose',
            flowchart: {
                useMaxWidth: true,
                htmlLabels: true
            }
        });
        
        // 错误处理
        mermaid.parseError = function(err, hash) {
            console.error('Mermaid解析错误:', err);
            const container = document.querySelector('.mermaid-container');
            container.innerHTML = '<div class="error-message">图表渲染失败: ' + err + '</div>';
        };
    </script>
</body>
</html>`;
  },

  /**
   * 生成画布HTML
   */
  generateCanvasHTML(canvasId, title, description, charts, theme) {
    // 处理每个图表的mermaid代码
    const processedCharts = charts.map(chart => ({
      ...chart,
      processedCode: this.processMermaidCode(chart.mermaidCode)
    }));
    
    const chartsHTML = processedCharts.map((chart, index) => `
        <div class="chart-section">
            <h2>${this.escapeHtml(chart.title || `图表 ${index + 1}`)}</h2>
            ${chart.description ? `<p class="chart-description">${this.escapeHtml(chart.description)}</p>` : ''}
            <div class="chart-container">
                <pre class="mermaid">
${chart.processedCode}
                </pre>
            </div>
        </div>`).join('');

    return `<!DOCTYPE html>
<html lang="zh-CN">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>${this.escapeHtml(title)}</title>
    <script src="https://cdn.jsdelivr.net/npm/mermaid@10/dist/mermaid.min.js"></script>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
            margin: 0;
            padding: 20px;
            background: ${theme === 'dark' ? '#1e1e1e' : '#f5f5f5'};
            color: ${theme === 'dark' ? '#e0e0e0' : '#333333'};
            line-height: 1.6;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
        }
        
        .canvas-header {
            margin-bottom: 40px;
            padding-bottom: 20px;
            border-bottom: 2px solid ${theme === 'dark' ? '#404040' : '#e0e0e0'};
        }
        
        .canvas-header h1 {
            font-size: 32px;
            margin-bottom: 15px;
            color: ${theme === 'dark' ? '#ffffff' : '#2c3e50'};
        }
        
        .canvas-description {
            font-size: 16px;
            padding: 15px;
            background: ${theme === 'dark' ? '#2d2d2d' : '#ffffff'};
            border-radius: 8px;
            border-left: 4px solid ${theme === 'dark' ? '#4CAF50' : '#2196F3'};
        }
        
        .chart-section {
            margin-bottom: 50px;
            background: ${theme === 'dark' ? '#2d2d2d' : '#ffffff'};
            border-radius: 12px;
            padding: 30px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .chart-section h2 {
            font-size: 24px;
            margin-bottom: 10px;
            color: ${theme === 'dark' ? '#ffffff' : '#333333'};
            border-bottom: 2px solid ${theme === 'dark' ? '#404040' : '#f0f0f0'};
            padding-bottom: 10px;
        }
        
        .chart-description {
            font-size: 14px;
            color: ${theme === 'dark' ? '#b0b0b0' : '#666666'};
            margin-bottom: 20px;
            padding: 10px;
            background: ${theme === 'dark' ? '#1e1e1e' : '#f8f9fa'};
            border-radius: 5px;
        }
        
        .chart-container {
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 300px;
            padding: 20px;
            background: ${theme === 'dark' ? '#1e1e1e' : '#fafafa'};
            border-radius: 8px;
            border: 1px solid ${theme === 'dark' ? '#404040' : '#e0e0e0'};
            overflow-x: auto;
        }
        
        .mermaid {
            width: 100%;
            height: auto;
        }
        
        /* 分隔线 */
        .chart-section:not(:last-child)::after {
            content: '';
            display: block;
            margin-top: 40px;
            height: 1px;
            background: ${theme === 'dark' ? '#404040' : '#e0e0e0'};
        }
        
        /* 响应式设计 */
        @media (max-width: 768px) {
            .container {
                padding: 10px;
            }
            
            .chart-section {
                padding: 20px;
            }
            
            .canvas-header h1 {
                font-size: 24px;
            }
            
            .chart-section h2 {
                font-size: 20px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="canvas-header">
            <h1>🎨 ${this.escapeHtml(title)}</h1>
            ${description ? `<div class="canvas-description">${this.escapeHtml(description)}</div>` : ''}
        </div>
        
        ${chartsHTML}
    </div>
    
    <script>
        // 初始化Mermaid
        mermaid.initialize({ 
            startOnLoad: true,
            theme: '${theme}',
            securityLevel: 'loose',
            flowchart: {
                useMaxWidth: true,
                htmlLabels: true,
                curve: 'basis'
            }
        });
        
        // 错误处理
        mermaid.parseError = function(err, hash) {
            console.error('Mermaid解析错误:', err);
        };
    </script>
</body>
</html>`;
  },

  /**
   * 保存HTML文件
   */
  async saveHTML(fileName, htmlContent) {
    const tempDir = path.join(os.homedir(), '.promptx', 'temp');
    if (!fs.existsSync(tempDir)) {
      fs.mkdirSync(tempDir, { recursive: true });
    }
    
    const filePath = path.join(tempDir, fileName);
    
    await new Promise((resolve, reject) => {
      fs.writeFile(filePath, htmlContent, 'utf8', (err) => {
        if (err) reject(err);
        else resolve();
      });
    });
    
    return filePath;
  },

  /**
   * HTML转义函数
   */
  escapeHtml(text) {
    if (!text) return '';
    const map = {
      '&': '&amp;',
      '<': '&lt;',
      '>': '&gt;',
      '"': '&quot;',
      "'": '&#039;'
    };
    return text.replace(/[&<>"']/g, m => map[m]);
  },

  /**
   * 处理Mermaid代码中的特殊字符
   */
  processMermaidCode(code) {
    if (!code) return '';
    
    // 处理@符号，将其替换为引号包裹的形式
    // 例如: @tool://name 变成 "tool://name"
    let processedCode = code.replace(/@(\w+:\/\/[^\s\]]+)/g, '"$1"');
    
    // 处理路径中的特殊字符
    // 将路径用引号包裹
    processedCode = processedCode.replace(/(\[)([^"\[\]]*~[^"\[\]]*)([\]\[])/g, '$1"$2"$3');
    
    // 确保所有包含特殊字符的节点文本都被引号包裹
    processedCode = processedCode.replace(/(\[)([^"\[\]]*[/@.][^"\[\]]*)(])/g, function(match, p1, p2, p3) {
      // 如果还没有被引号包裹
      if (!p2.startsWith('"') && !p2.endsWith('"')) {
        return p1 + '"' + p2 + '"' + p3;
      }
      return match;
    });
    
    return processedCode;
  }
};